/*
 * linux/include/linux/exchnd.h
 *
 * Copyright (C) 2013 Advanced Driver Information Technology GmbH
 * Written by Matthias Weise (mweise@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.   See the
 * GNU General Public License for more details.
 *
 */

#ifndef __EXCHND_H__
#define __EXCHND_H__
#include <linux/types.h>

#include <signal.h>

#define EXCHND_MAGIC_IOC 0xec

/*
 * Exception handler module identifiers
 */
enum exchnd_modules {
	EHM_NONE = 0,
	EHM_BACKTRACE,
	EHM_BACKTRACE_ALL_THREADS,
	EHM_CGROUPS,
	EHM_CORE_DUMP, /* UNUSED */
	EHM_CPU_USAGE,
	EHM_FAULT_ADDRESS,
	EHM_FS_STATE,
	EHM_HIST_SYSCALLS,
	EHM_HIST_TASKSWITCHES,
	EHM_LRU_MEM_PAGES,
	EHM_MEMORY_DUMP,
	EHM_MEMORY_MAP,
	EHM_MEMORY_USAGE,
	EHM_PROCESSOR_REGISTERS,
	EHM_PROCESS_LIST,
	EHM_SCHED_INF,
	EHM_STACK_DUMP,
	EHM_SYSTEM_INFO,
	EHM_THREAD_LIST,
	/* Values for application specific modules */
	EHM_APPLICATION_SPECIFIC1,
	EHM_APPLICATION_SPECIFIC2,
	EHM_APPLICATION_SPECIFIC3,
	EHM_APPLICATION_SPECIFIC4,
	EHM_APPLICATION_SPECIFIC5,
	/* Action modules executed after data collection */
	EHM_ACTION_START,
	EHM_SYS_RESTART,
	/* End marker for number of types */
	EHM_LAST_ELEMENT
};

/*
 * Module strings
 *
 * Defined as macro to have them defined in one place for kernel and
 * user space.
 */
#define EXCHND_MODULE_STRINGS { \
	"NONE", \
	"BACKTRACE", \
	"BACKTRACE_ALL_THREADS", \
	"CGROUPS", \
	"CORE_DUMP", \
	"CPU_USAGE", \
	"FAULT_ADDRESS", \
	"FS_STATE", \
	"HIST_SYSCALLS", \
	"HIST_TASKSWITCHES", \
	"LRU_MEM_PAGES", \
	"MEMORY_DUMP", \
	"MEMORY_MAP", \
	"MEMORY_USAGE", \
	"PROCESSOR_REGISTERS", \
	"PROCESS_LIST", \
	"SCHED_INF", \
	"STACK_DUMP",\
	"SYSTEM_INFO",\
	"THREAD_LIST", \
	"APPLICATION_SPECIFIC1", \
	"APPLICATION_SPECIFIC2", \
	"APPLICATION_SPECIFIC3", \
	"APPLICATION_SPECIFIC4", \
	"APPLICATION_SPECIFIC5", \
	"ACTION_START", \
	"SYS_RESTART" \
}

/*
 * Exception handler trigger identifiers
 */
enum exchnd_triggers {
	ET_NONE = 0,
	ET_EXTERNAL,
	ET_HALT,
	ET_KERNEL_OOPS,
	ET_KERNEL_PANIC,
	ET_ON_DEMAND,
	ET_OOM,
	ET_POWER_OFF,
	ET_PROCESS_EXIT,
	ET_PROCESS_SIGNAL,
	ET_RESTART,
	ET_LAST_ELEMENT
};

/*
 * Trigger strings
 *
 * Defined as macro to have them defined in one place for kernel and
 * user space.
 */
#define EXCHND_TRIGGER_STRINGS { \
	"NONE", \
	"EXTERNAL", \
	"HALT", \
	"KERNEL_OOPS", \
	"KERNEL_PANIC", \
	"ON_DEMAND", \
	"OOM", \
	"POWER_OFF", \
	"PROCESS_EXIT", \
	"PROCESS_SIGNAL", \
	"RESTART" \
}

/*
 * Signal strings
 *
 * Defined as macro to have them defined in one place for kernel and
 * user space.
 */
#define EXCHND_SIGNAL_STRINGS { \
	"NONE", \
	"SIGHUP", \
	"SIGINT", \
	"SIGQUIT", \
	"SIGILL", \
	"SIGTRAP", \
	"SIGABRT", \
	"SIGBUS", \
	"SIGFPE", \
	"SIGKILL", \
	"SIGUSR1", \
	"SIGSEGV", \
	"SIGUSR2", \
	"SIGPIPE", \
	"SIGALRM", \
	"SIGTERM", \
	"SIGSTKFLT", \
	"SIGCHLD", \
	"SIGCONT", \
	"SIGSTOP", \
	"SIGTSTP", \
	"SIGTTIN", \
	"SIGTTOU", \
	"SIGURG", \
	"SIGXCPU", \
	"SIGXFSZ", \
	"SIGVTALRM", \
	"SIGPROF", \
	"SIGWINCH", \
	"SIGPOLL", \
	"SIGPWR", \
	"SIGSYS"\
}

enum exchnd_conf {
	EC_NO_MODULES = 1,
	EC_SAFETY,
	EC_DEBUG,
	EC_VERBOSE,
	EC_RBCM,
#define EC_RBCM EC_RBCM
	EC_DN_NORESTART,
#define EC_DN_NORESTART EC_DN_NORESTART
	EC_DN,
#define EC_DN EC_DN
	EC_LAST_ELEMENT
};

enum exchnd_signal_conf {
	ESC_NO_MODULES = 1,
	ESC_SAFETY,
	ESC_DEBUG,
	ESC_VERBOSE,
	ESC_ALL_MODULES,
	ESC_RBCM_NORESTART,
#define ESC_RBCM_NORESTART ESC_RBCM_NORESTART
	ESC_RBCM_RESTART,
#define ESC_RBCM_RESTART ESC_RBCM_RESTART
	ESC_DN_NORESTART,
#define ESC_DN_NORESTART ESC_DN_NORESTART
	ESC_DN_RESTART,
#define ESC_DN_RESTART ESC_DN_RESTART
	ESC_LAST_ELEMENT
};

enum exchnd_pexit_conf {
	EPEC_NO_MODULES = 1,
	EPEC_SAFETY,
	EPEC_DEBUG,
	EPEC_VERBOSE,
	EPEC_ALL_MODULES,
	EPEC_LAST_ELEMENT
};

/*
 * Fatality of an exception
 */
enum exchand_fatality {
	FT_FATAL,
	FT_NON_FATAL
};

/*
 * Flags that show the state of the exception handler message
 */
struct exc_msg_flags {
	unsigned collected:1;		/* 1 = data collected in kernel,
					   0 = no data was collected */
	unsigned internal:1;		/* 1 = data for internal purposes
					   0 = data for the output channels */
	unsigned addition_needed:1;	/* 1 = additional collection needed,
					   0 = data is complete */
};

/*
 * Control interface of the module for the exception handler daemon
 */
#define EXCHND_VERSION   0x214 /* v2.14 */

enum exchnd_conf_type {
	EXCHND_CONF_TRIGGER,
	EXCHND_CONF_SIGNAL,
	EXCHND_CONF_DEFAULT_SIGNAL,
	EXCHND_CONF_DEFAULT_EXIT,
	EXCHND_CONF_TRIGGER_SET,
	EXCHND_CONF_SIGNAL_SET,
	EXCHND_CONF_SIGHDL_MASK,
	EXCHND_CONF_MODULE_ENABLE,
	EXCHND_CONF_MODULE_DISABLE,
	EXCHND_CONF_MODULE_HIST_SIZE,
	EXCHND_CONF_MODULE_HIST_PID
};

struct exchnd_conf_param {
	enum exchnd_conf_type conf_type;
	union {
		enum exchnd_modules modules[EHM_LAST_ELEMENT + 1];
		enum exchnd_modules triggers[ET_LAST_ELEMENT][EHM_LAST_ELEMENT];
		enum exchnd_modules signal[SIGSYS+1][EHM_LAST_ELEMENT];
		unsigned int sig_handled_mask;
		unsigned int pid;
		unsigned int size;
	};
};

#define EXH_PATH_MAX 4096
struct exchnd_conf_app_spec {
	enum exchnd_modules module;
	char lib_path[EXH_PATH_MAX];
};

#define MAX_FILTER_LENGTH 255

struct exchnd_conf_filter {
	char id[MAX_FILTER_LENGTH];
	int is_kernel;
	int add;
};

#define EXH_ON_DEMAND_MSG_LEN 256
struct exchnd_on_demand {
	pid_t pid;
	int use_def;
	enum exchnd_modules modules[EHM_LAST_ELEMENT];
	char msg[EXH_ON_DEMAND_MSG_LEN];
};

/* IOCTL_EXCHND_VERSION - version of this driver */
#define IOCTL_EXCHND_VERSION				\
	_IOR(EXCHND_MAGIC_IOC, 0, unsigned long)

/* IOCTL_EXCHND_ON_DEMAND - raise an ON_DEMAND exception */
#define IOCTL_EXCHND_ON_DEMAND				\
	_IOW(EXCHND_MAGIC_IOC, 1, struct exchnd_on_demand*)

/* IOCTL_EXCHND_CONFIGURATION_GET
 *  - get a list of exception handler modules per trigger */
#define IOCTL_EXCHND_CONFIGURATION_GET				\
	_IOWR(EXCHND_MAGIC_IOC, 2, struct exchnd_conf_param*)

/* IOCTL_EXCHND_CONFIGURATION_SET
 *  - set a list of exception handler modules for one trigger */
#define IOCTL_EXCHND_CONFIGURATION_SET				\
	_IOW(EXCHND_MAGIC_IOC, 3, struct exchnd_conf_param*)

/* IOCTL_EXCHND_DAEMON_READY
 *  - indicate that the daemon has finished the output */
#define IOCTL_EXCHND_DAEMON_READY				\
	_IOW(EXCHND_MAGIC_IOC, 4, unsigned long)

/* IOCTL_EXCHND_MODIFY_FILTER
 *  - add/remove process to the process filter white list */
#define IOCTL_EXCHND_MODIFY_FILTER				\
	_IOW(EXCHND_MAGIC_IOC, 5, struct exchnd_conf_filter*)

/* IOCTL_EXCHND_APP_SPECIFIC
 *  - add/remove process to the process filter white list */
#define IOCTL_EXCHND_APP_SPECIFIC				\
	_IOW(EXCHND_MAGIC_IOC, 6, struct exchnd_conf_app_spec*)

/* IOCTL_EXCHND_RECOVERY
 *  - ask for buffer read pointer recovery */
#define IOCTL_EXCHND_RECOVERY				\
	_IO(EXCHND_MAGIC_IOC, 7)

/* IOCTL_EXCHND_MODULES
 * - Allow to enable/disable collectors modules
 */
#define IOCTL_EXCHND_MODULES					\
	_IOW(EXCHND_MAGIC_IOC, 8, struct exchnd_conf_param*)

/*
 * Header for the exception message send from kernel to user space
 *
 * This message message header contains the meta data for the data collected
 * in the kernel for an information type (done by an exception handler
 * module). This header contains all meta data needed to process the
 * collected data
 */
struct exchnd_message_header {
	unsigned int length;		/* Length of the collected data */
	enum exchnd_modules type;	/* Type of collected data */
	enum exchnd_triggers trigger;	/* Trigger of the exception */
	pid_t pid;                      /* ID of process causing exception */
	unsigned int seq_num;		/* Sequence no. of the msg per mod */
	struct exc_msg_flags flags;	/* Flags for the exception message */
	unsigned int magic;		/* Magic number for recovery */
	unsigned int sig;		/* Signal to be handled if relevant */
	unsigned int reserved3;		/* Reserved for future use */
	unsigned int reserved4;		/* Reserved for future use */
};


#endif
